<?php namespace App\Controllers\Api\Customer\Payment\PayPal;

use App\Controllers\PrivateController;
use App\Libraries\Settings;
use App\Models\AppsModel;
use App\Models\DepositMethodsModel;
use App\Models\PlansExtraModel;
use App\Models\SubscribesExtraModel;
use App\Models\SubscribesModel;
use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\Database\Exceptions\DatabaseException;

class CreatePaymentExtraRequest extends PrivateController
{
    protected $paypalUrl = 'https://api-m.paypal.com/v2/checkout/orders'; // Chuyển sang live

    public function index(): ResponseInterface
    {
        if (!$this->validate($this->create_validation_type())) {
            return $this->respond(["message" => $this->validator->getErrors()], 400);
        }

        $subscribe_uid = esc($this->request->getVar("uid"));
        log_message('debug', 'Received UID: ' . $subscribe_uid);
        $plan_extra_id = esc($this->request->getVar("plan_extra_id"));
        log_message('debug', 'Received plan_extra_id: ' . $plan_extra_id);
    
        $db = \Config\Database::connect();
        $db->transStart();

        try {
            $app = $this->getAppFromSubscribe($subscribe_uid);
            $this->checkActiveSubscription($app["id"]);
            $extraPlan = $this->getPlanExtra($plan_extra_id);
            $method = $this->getPayPalMethod();

            $order = $this->createPayPalOrder($method, $extraPlan, $app["id"]);

            if (isset($order['links'][1]['href'])) {
                $db->transComplete();
                return $this->respond(["url" => $order['links'][1]['href']], 200);
            } else {
                throw new DatabaseException("PayPal order creation failed");
            }
        } catch (\Exception $e) {
            $db->transRollback();
            return $this->respond(["message" => $e->getMessage()], 400);
        }
    }

    private function getAppFromSubscribe($subscribe_uid)
    {
        $subscribes = new SubscribesModel();
        $subscribe = $subscribes
            ->where('uid', $subscribe_uid)
            ->select("app_id")
            ->first();
      
        $projects = new AppsModel();
        $app = $projects->where([
            "id" => $subscribe['app_id'],
            "user" => $this->userId,
            "deleted_at" => 0
        ])->select("id")->first();

        if (!$app) {
            throw new DatabaseException(lang("Message.message_14"));
        }

        return $app;
    }

    private function checkActiveSubscription($appId)
    {
        $subscribes_extra = new SubscribesExtraModel();
        $is_active = $subscribes_extra->where([
            "app_id" => $appId,
            "is_disable" => 0
        ])->countAllResults();
    }
  
    private function getPlanExtra($plan_extra_id)
    {
        $plansExtra = new PlansExtraModel();
        $extraPlan = $plansExtra->where([
            "id" => $plan_extra_id,
            "deleted_at" => 0
        ])->first();

        if ($extraPlan['deleted_at'] != 0) {
            throw new DatabaseException(lang("Message.message_99"));
        }

        return $extraPlan;
    }

    private function getPayPalMethod()
    {
        $methods = new DepositMethodsModel();
        $method = $methods->where([
            "name" => "PayPal",
            "status" => 1
        ])->first();

        if (!$method) {
            throw new DatabaseException(lang("Message.message_84"));
        }

        return $method;
    }

    private function createPayPalOrder($method, $extraPlan, $appId)
    {
        $settings = new Settings();
        $frontUrl = rtrim($settings->get_config("site_url"), '/') . '/';

        $returnUrl = base_url('public/ipn/paypal/capture');
        $cancelUrl = $frontUrl . 'private/profile/subscribe';

        $clientId = $method["api_value_1"];
        $clientSecret = $method["api_value_2"];
        $auth = base64_encode("$clientId:$clientSecret");
      
        // Lấy mã tiền tệ từ bảng settings
        $currencyCode = $settings->get_config("currency_code");

        log_message('debug', 'PayPal Auth: ' . $auth);
        log_message('debug', 'Currency Code: ' . $currencyCode);

        $client = service('curlrequest');
        $response = $client->post($this->paypalUrl, [
            'headers' => [
                'Authorization' => "Basic $auth",
                'Content-Type' => 'application/json'
            ],
            'json' => [
                'intent' => 'CAPTURE',
                'purchase_units' => [[
                    'amount' => [
                        'currency_code' => $currencyCode,
                        'value' => $extraPlan["price"]
                    ],
                    'custom_id' => 'EXTRA_' . $extraPlan["id"] . '_' . $this->userId . '_' . $appId,
                ]],
                'application_context' => [
                    'return_url' => $returnUrl,
                    'cancel_url' => $cancelUrl
                ]
            ]
        ]);

        return json_decode($response->getBody(), true);
    }

    private function create_validation_type(): array
    {
        return [
            "plan_extra_id" => [
                "label" => lang("Fields.field_160"),
                "rules" => "required|numeric"
            ],
        ];
    }
}
